globals [
  ppatches  ;; like "patches", but excludes the top row and rightmost column
  lattice   ;; only those ppatches where both pxcor and pycor are even
]

to setup
  ca
  ask patches
    [ set pcolor white ]
  set ppatches patches with [pxcor < screen-edge-x and
                             pycor < screen-edge-y]
  set lattice ppatches with [pxcor mod 2 = 0 and pycor mod 2 = 0]
  ask ppatches
    [ if random-float 100 < density
        [ set pcolor black ] ]
end

to go
  ask lattice [ do-rule  1 ]  ;; propagation
  ask lattice [ do-rule -1 ]  ;; collision
end

to go-reverse  ;; applying rules to the lattice in reverse order reverses the system
  ask lattice [ do-rule -1 ]  ;; collision
  ask lattice [ do-rule  1 ]  ;; propagation
end

;; grid = 1 if even lattice, grid = -1 if odd lattice
to do-rule [grid]
  let a self
  let b ppatchat (- grid) 0
  let c ppatchat 0 grid
  let d ppatchat (- grid) grid
  
  ifelse (pcolor-of a) != (pcolor-of b) and
         (pcolor-of c) != (pcolor-of d)
    [ swap-pcolor a b
      swap-pcolor c d ]
    [ swap-pcolor a d
      swap-pcolor b c ]
end

to swap-pcolor [p1 p2]
  let temp pcolor-of p1
  set (pcolor-of p1) pcolor-of p2
  set (pcolor-of p2) temp
end

to draw-circle
  if mouse-down?
    [ ask patch (round mouse-xcor) (round mouse-ycor)
        [ ask patches in-radius-nowrap radius
            [ if pxcor < screen-edge-x and
                 pycor < screen-edge-y
                [ set pcolor black ] ] ] ]
end

;;;
;;; what follows is support code for operating on an even-sized grid
;;;

to-report ppatchat [xoff yoff]  ;; patch procedure
  report patch-at adjust-x-offset xoff
                  adjust-y-offset yoff
end

to-report adjust-x-offset [xoff]  ;; patch procedure
  ;; if we landed in the rightmost column, go one patch farther
  ifelse pxcor-of patch-at xoff 0 = screen-edge-x
    [ report xoff * 2 ] ;; changes 1 to 2, -1 to -2
    [ report xoff ]
end

to-report adjust-y-offset [yoff]  ;; patch procedure
  ;; if we landed in the top row, go one patch farther
  ifelse pycor-of patch-at 0 yoff = screen-edge-y
    [ report yoff * 2 ] ;; changes 1 to 2, -1 to -2
    [ report yoff ]
end


; *** NetLogo Model Copyright Notice ***
;
; This model was created as part of the project:
; PARTICIPATORY SIMULATIONS: NETWORK-BASED DESIGN FOR SYSTEMS LEARNING IN
; CLASSROOMS.  The project gratefully acknowledges the support of the
; National Science Foundation (REPP program) -- grant number REC #9814682.
;
; Copyright 2002 by Uri Wilensky.  Updated 2003.  All rights reserved.
;
; Permission to use, modify or redistribute this model is hereby granted,
; provided that both of the following requirements are followed:
; a) this copyright notice is included.
; b) this model will not be redistributed for profit without permission
;    from Uri Wilensky.
; Contact Uri Wilensky for appropriate licenses for redistribution for
; profit.
;
; To refer to this model in academic publications, please use:
; Wilensky, U. (2002).  NetLogo Lattice Gas Automaton model.
; http://ccl.northwestern.edu/netlogo/models/LatticeGasAutomaton.
; Center for Connected Learning and Computer-Based Modeling,
; Northwestern University, Evanston, IL.
;
; In other publications, please use:
; Copyright 1998 by Uri Wilensky.  All rights reserved.  See
; http://ccl.northwestern.edu/netlogo/models/LatticeGasAutomaton
; for terms of use.
;
; *** End of NetLogo Model Copyright Notice ***
@#$#@#$#@
GRAPHICS-WINDOW
298
10
630
363
80
80
2.0
1
10
1
1
1
0

CC-WINDOW
5
377
639
472
Command Center

BUTTON
179
41
280
74
NIL
setup
NIL
1
T
OBSERVER
T
NIL

BUTTON
20
89
144
135
NIL
go
T
1
T
OBSERVER
T
NIL

SLIDER
7
41
179
74
density
density
0.0
100.0
20.0
1.0
1
%

BUTTON
180
164
281
197
NIL
draw-circle
T
1
T
OBSERVER
NIL
NIL

SLIDER
8
164
180
197
radius
radius
0
20
10
1
1
NIL

BUTTON
147
89
271
135
reverse
go-reverse
T
1
T
OBSERVER
T
NIL

@#$#@#$#@
WHAT IS IT?
-----------
This model demonstrates circular wave propagation using a cellular automaton on a square grid.  The behavior of the waves approximates the Navier-Stokes equation, a well established fluid dynamics equation discovered in 1823.

In the model, space is divided in to cells which are occupied by the gas particles.  Each particle has the same mass and absolute velocity (each particle will move only a single cell at each time step).  Space is broken up into neighborhoods of 2x2 cell squares.  Collisions occur when multiple particles are in the same square and occur instantaneously, conserving both mass and momentum.

The model is implemented using a Margolus neighborhood of 2x2 cell squares in which the particles belong to two separate spacetime sublattices, propagation and collision, (sometimes refered to as "even" and "odd" lattices), which evolve independently of each other.


HOW IT WORKS
------------
The CA can be summarized with the following core rules:

|    W W  ->  W W                B W  ->  W W            B W  ->  W B
|    W W      W W                W W      W B            W B      B W
|
|    B W  ->  W B                W B  ->  B B            B B  ->  B B
|    B W      W B                B B      B W            B B      B B

It applies these rules to the even lattice of 2x2 squares followed by the odd lattice of 2x2 squares.  Specifically, this means that the top left patch of four applies one of the above rules, then the bottom right patch of four does the same.  To run the model in reverse, we simply switch the order of those two operations.  No patch gets changed more then twice per tick.  That means that each patch sees only one possible even rule and one possible odd rule each iteration.


HOW TO USE IT
-------------
The basic controls for the model are:
SETUP - Sets up screen with a given percentage of particles
DENSITY - Percentage of particles in gas
GO - Run the model
REVERSE - Run the model in reverse

These controls let you "paint" an initial setup on the screen using the mouse:
DRAW-CIRCLE - Clicking any location on the screen creates a solid circle of particles
RADIUS - Controls the radius of the circles


THINGS TO NOTICE
----------------
Drawing a circle in the center of the screen and clicking GO creates a circular wave that travels through the lattice.  Once the wave reaches the edges of the screen, it wraps around the sides and causes the wave to collide with itself. You may stop the model by clicking GO again, and reverse the system by clicking REVERSE.  The wave will now implode and eventually return to its original starting state.

How does the density of the center of the wave vary over time? Why?

Why does the model appear to act the same when run forwards and backwards?


THINGS TO TRY
-------------
How does the density of the gas effect the propagation of the wave?  Try running the model with various DENSITY settings. Why do certain densities impede the propagation of waves?

What happens when you create multiple compressions in the gas and run the model?


EXTENDING THE MODEL
-------------------
Can you create obstacles that deflect the movement of particles?

This particular model is known as the HPP model.  The HPP model is very limited. The FHP (Frisch, Hasslacher and Pomeau) model was invented in the mid-eighties in order to improve the accuracy of the HPP model.  The underlying rules of the FHP model are similar to that of the HPP model except that the FHP model has a symmetry-group order of six.  The hexagonal lattice allows for more advanced modeling, such as hydrodynamical simulations.  Can you write a model that emulates a hexagonal lattice in NetLogo?


NETLOGO FEATURES
----------------
In order for the algorithm to operate correctly, the size of the lattice must be even in both dimensions.  However, the current version of NetLogo only supports odd sized grids.  Therefore, it was necessary to add special code to this model to ignore the top row and rightmost column of the grid, effectively converting an odd-sized grid into an even-sized one.  In the code, the "ppatches" variable holds all the patches except the excluded row and column, and the "ppatchat" procedure is like the built-in "patch-at" except for skipping that row and column.

See also the "Even Grid Example" in the Code Examples section of the Models Library.


CREDITS AND REFERENCES
-----------------------
Thanks to Ethan Bakshy for his help with this model.

For more information about lattice gas automata, see:

J. Hardy, Y. Pomeau & O. de Pazzis, Time evolution of two-dimensional model system. I. Invariant states and time correlation functions, J. Math. Phys. 14 (1973), pp. 1746-1759.

U. Frisch, B. Hasslacher & Y. Pomeau, Lattice-gas automata for the Navier-Stokes equation, Phys. Rev. Lett. 56 (1986), pp. 1505-1508. 

T. Toffoli and N. Margolus. 1987. "Cellular Automata Machines: A New Environment for Modeling". 

Wolfram, S. 2002. A New Kind of Science. Wolfram Media Inc.  Champaign, IL.

To refer to this model in academic publications, please use: Wilensky, U. (2002).  NetLogo Lattice Gas Automaton model. http://ccl.northwestern.edu/netlogo/models/LatticeGasAutomaton.  Center for Connected Learning and Computer-Based Modeling, Northwestern University, Evanston, IL.

In other publications, please use: Copyright 2002 by Uri Wilensky.  All rights reserved.  See http://ccl.northwestern.edu/netlogo/models/LatticeGasAutomaton for terms of use.
@#$#@#$#@
default
true
0
Polygon -7566196 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7566196 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7566196 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7566196 true true 150 285 285 225 285 75 150 135
Polygon -7566196 true true 150 135 15 75 150 15 285 75
Polygon -7566196 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7566196 true true 96 182 108
Circle -7566196 true true 110 127 80
Circle -7566196 true true 110 75 80
Line -7566196 true 150 100 80 30
Line -7566196 true 150 100 220 30

butterfly
true
0
Polygon -7566196 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7566196 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7566196 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7566196 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7566196 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7566196 true true 47 195 58
Circle -7566196 true true 195 195 58

circle
false
0
Circle -7566196 true true 30 30 240

circle 2
false
0
Circle -7566196 true true 16 16 270
Circle -16777216 true false 46 46 210

cow
false
0
Polygon -7566196 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7566196 true true 73 210 86 251 62 249 48 208
Polygon -7566196 true true 25 114 16 195 9 204 23 213 25 200 39 123

face happy
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7566196 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7566196 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7566196 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7566196 true true 60 15 75 300
Polygon -7566196 true true 90 150 270 90 90 30
Line -7566196 true 75 135 90 135
Line -7566196 true 75 45 90 45

flower
false
0
Polygon -11352576 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7566196 true true 85 132 38
Circle -7566196 true true 130 147 38
Circle -7566196 true true 192 85 38
Circle -7566196 true true 85 40 38
Circle -7566196 true true 177 40 38
Circle -7566196 true true 177 132 38
Circle -7566196 true true 70 85 38
Circle -7566196 true true 130 25 38
Circle -7566196 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -11352576 true false 189 233 219 188 249 173 279 188 234 218
Polygon -11352576 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7566196 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7566196 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7566196 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7566196 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7566196 true 150 0 150 300

pentagon
false
0
Polygon -7566196 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7566196 true true 110 5 80
Polygon -7566196 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7566196 true true 127 79 172 94
Polygon -7566196 true true 195 90 240 150 225 180 165 105
Polygon -7566196 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7566196 true true 135 90 165 300
Polygon -7566196 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7566196 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7566196 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7566196 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7566196 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7566196 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7566196 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7566196 true true 30 30 270 270

square 2
false
0
Rectangle -7566196 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7566196 true true 60 270 150 0 240 270 15 105 285 105
Polygon -7566196 true true 75 120 105 210 195 210 225 120 150 75

target
false
0
Circle -7566196 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7566196 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7566196 true true 120 120 60

tree
false
0
Circle -7566196 true true 118 3 94
Rectangle -6524078 true false 120 195 180 300
Circle -7566196 true true 65 21 108
Circle -7566196 true true 116 41 127
Circle -7566196 true true 45 90 120
Circle -7566196 true true 104 74 152

triangle
false
0
Polygon -7566196 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7566196 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7566196 true true 4 45 195 187
Polygon -7566196 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7566196 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7566196 false true 24 174 42
Circle -7566196 false true 144 174 42
Circle -7566196 false true 234 174 42

turtle
true
0
Polygon -11352576 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -11352576 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -11352576 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -11352576 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -11352576 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7566196 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7566196 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7566196 true 150 285 150 15
Line -7566196 true 15 150 285 150
Circle -7566196 true true 120 120 60
Line -7566196 true 216 40 79 269
Line -7566196 true 40 84 269 221
Line -7566196 true 40 216 269 79
Line -7566196 true 84 40 221 269

x
false
0
Polygon -7566196 true true 270 75 225 30 30 225 75 270
Polygon -7566196 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 2.1beta4
@#$#@#$#@
setup
ask patch 0 0
  [ ask patches in-radius 15
      [ set pcolor black ] ]
repeat 15 [ go ]
@#$#@#$#@
@#$#@#$#@
